ColorConfiguratorRenderElement = {}

local ColorConfiguratorRenderElement_mt = Class(ColorConfiguratorRenderElement, RenderElement)

function ColorConfiguratorRenderElement.new(target, custom_mt)
	local self = RenderElement.new(target, custom_mt or ColorConfiguratorRenderElement_mt)

	self.mouseDown = false
	self.restorePosition = true

	self.objectPath = "0|0"

	self.stepSize = 0.07
	self.transMin = 0.93
	self.transMax = 2.47

	self.rotSpeed = 3.5
	self.rotMin = math.rad(-55)
	self.rotMax = math.rad(30)

	return self
end

function ColorConfiguratorRenderElement:getIsActive()
	local baseActive = ColorConfiguratorRenderElement:superClass().getIsActive(self)

	return baseActive and self.sceneNode ~= nil and self.cameraNode ~= nil and self.rotateNode ~= nil
end

function ColorConfiguratorRenderElement:createOverlay(node, failedReason, args)
	local sceneRoot = self:getSceneRoot()
	local sceneNode = I3DUtil.indexToObject(sceneRoot, self.objectPath)
	local cameraNode = I3DUtil.indexToObject(sceneRoot, self.cameraPath)

	self.sceneNode = sceneNode
	self.cameraNode = cameraNode
	self.rotateNode = getParent(cameraNode)

	local _, _, zPos = getTranslation(self.cameraNode)
	local xRot, yRot, _ = getRotation(self.rotateNode)

	self.defaultCameraPosition = zPos
	self.defaultRotation = {xRot, yRot}

	if self.restorePosition then
		if self.lastCameraPosition ~= nil then
			setTranslation(self.cameraNode, 0, 0, self.lastCameraPosition)
		end

		if self.lastRotation ~= nil then
			setRotation(self.rotateNode, self.lastRotation[1], self.lastRotation[2], 0)
		end
	end

	ColorConfiguratorRenderElement:superClass().createOverlay(self, node, failedReason, args)
end

function ColorConfiguratorRenderElement:destroyScene()
	ColorConfiguratorRenderElement:superClass().destroyScene(self)

	self.sceneNode = nil
	self.cameraNode = nil
	self.rotateNode = nil
end

function ColorConfiguratorRenderElement:mouseEvent(posX, posY, isDown, isUp, button, eventUsed)
	if self:getIsActive() then
		if ColorConfiguratorRenderElement:superClass().mouseEvent(self, posX, posY, isDown, isUp, button, eventUsed) then
			eventUsed = true
		end

		if self.mouseDown and isUp and button == Input.MOUSE_BUTTON_LEFT then
			eventUsed = true
			self.mouseDown = false
		end

		if not eventUsed then
			if GuiUtils.checkOverlayOverlap(posX, posY, self.absPosition[1], self.absPosition[2], self.absSize[1], self.absSize[2]) then
				eventUsed = true

				if isDown then
					if Input.isMouseButtonPressed(Input.MOUSE_BUTTON_WHEEL_UP) then
						self:updateCameraZoom(-self.stepSize)
					end

					if Input.isMouseButtonPressed(Input.MOUSE_BUTTON_WHEEL_DOWN) then
						self:updateCameraZoom(self.stepSize)
					end

					if button == Input.MOUSE_BUTTON_LEFT then
						if not self.mouseDown then
							self.lastMousePosX = posX
							self.lastMousePosY = posY
							self.mouseDown = true
						end
					elseif button == Input.MOUSE_BUTTON_MIDDLE then
						self:resetPosition()
					end
				end
			end
		end

		if self.mouseDown then
			eventUsed = true

			self:updateRotation((posY - self.lastMousePosY) * self.rotSpeed, (posX - self.lastMousePosX) * self.rotSpeed)

			self.lastMousePosX = posX
			self.lastMousePosY = posY
		end
	end

	return eventUsed
end

function ColorConfiguratorRenderElement:updateCameraZoom(offset)
	local _, _, z = getTranslation(self.cameraNode)

	z = MathUtil.clamp(z + offset, self.transMin, self.transMax)

	setTranslation(self.cameraNode, 0, 0, z)
	self:setRenderDirty()
	self.lastCameraPosition = z
end

function ColorConfiguratorRenderElement:updateRotation(offsetX, offsetY)
	local x, y, _ = getRotation(self.rotateNode)

	x, y = MathUtil.clamp(x + offsetX, self.rotMin, self.rotMax), y - offsetY

	setRotation(self.rotateNode, x, y, 0)
	self:setRenderDirty()
	self.lastRotation = {x, y}
end

function ColorConfiguratorRenderElement:updateShaderParameter(r, g, b, mat)
	if self:getIsActive() and getHasShaderParameter(self.sceneNode, "colorMat0") then
		setShaderParameter(self.sceneNode, "colorMat0", r, g, b, mat, false)
		self:setRenderDirty()
	end
end

function ColorConfiguratorRenderElement:resetPosition()
	setTranslation(self.cameraNode, 0, 0, self.defaultCameraPosition)
	setRotation(self.rotateNode, self.defaultRotation[1], self.defaultRotation[2], 0)

	self.lastCameraPosition = self.defaultCameraPosition
	self.lastRotation = self.defaultRotation

	self:setRenderDirty()
end